// Version 1.5.2

// Decode uplink function.
//
// Input is an object with the following fields:
// - bytes = Byte array containing the uplink payload, e.g. [255, 230, 255, 0]
// - fPort = Uplink fPort.
// - variables = Object containing the configured device variables.
//
// Output must be an object with the following fields:
// - data = Object representing the decoded payload.

function decodeUplink(input) {
  return{
    data: Decode(input.bytes, input.variables)
  };
}

const siTypes = {
    0: {name: "Undef"},
    1: {name: "InH₂O"},
    2: {name: "InHg"},
    3: {name: "ftH₂O"},
    4: {name: "mmH₂O"},
    5: {name: "mmHg"},
    6: {name: "psi"},
    7: {name: "bar"},
    8: {name: "mbar"},
    9: {name: "g/cm²"},
    10: {name: "kg/cm²"},
    11: {name: "Pa"},
    12: {name: "kPa"},
    13: {name: "torr"},
    14: {name: "atm"},
    32: {name: "°C"},
    33: {name: "°F"},
    34: {name: "°Rk"},
    35: {name: "K"},
    36: {name: "mV"},
    37: {name: "Ohm"},
    39: {name: "mA"},
    57: {name: "%"},
    58: {name: "V"},
    145: {name: "InH₂0 @60°F"},
    163: {name: "kΩ"},
    170: {name: "cmH₂0 @4°C"},
    171: {name: "mH₂0 @4°C"},
    172: {name: "cmHg"},
    173: {name: "lb/ft²"},
    174: {name: "hPa"},
    175: {name: "psia"},
    176: {name: "kg/m²"},
    177: {name: "ftH₂0 @4°C"},
    178: {name: "ftH₂0 @60°F"},
    179: {name: "mHg"},
    180: {name: "Mpsi"},
    181: {name: "oz/in²"},
    237: {name: "MPa"},
    238: {name: "InH₂0 @4°C"},
    239: {name: "mmH₂0 @4°C"},
    250: {name: "Not used"},
    251: {name: "None"},
    252: {name: "Unknown"},
    253: {name: "Special"},
  };
  
const varTypes = {
    0: { size: 4, name: "Current", defaultSI: 39},
    1: { size: 5, name: "PV",  defaultSI: 0},
    2: { size: 1, name: "Status",  defaultSI: 0},
    3: { size: 3, name: "UID",   defaultSI: 0},
    4: { size: 5, name: "PVMin",  defaultSI: 0 },
    5: { size: 5, name: "PVMax",  defaultSI: 0},
    6: { size: 4, name: "MoreStatus",  defaultSI: 0 },
    7: { size: 4, name: "Percent",  defaultSI: 57 },
    8: { size: 4, name: "UNDEF1",  defaultSI: 0 },
    9: { size: 4, name: "UNDEF2",  defaultSI: 0},
    10: { size: 1, name: "UNDEF3",  defaultSI: 0},
    11: { size: 4, name: "UNDEF4",  defaultSI: 0},
    12: { size: 4, name: "Battery",  defaultSI: 57},
    13: { size: 4, name: "UNDEF5",   defaultSI: 0},	
    14: { size: 2, name: "SelfState",  defaultSI: 0},
    15: { size: 2, name: "ConfGETBLOCK",  defaultSI: 0},
    16: { size: 13, name: "ConfDEVICEINFO",  defaultSI: 0},
    17: { size: 15, name: "ConfTRESHOLD",  defaultSI: 0},
    18: { size: 4, name: "ConfMEASURE",  defaultSI: 0},
    19: { size: 7, name: "ConfLORA",  defaultSI: 0},
    20: { size: 2, name: "ConfREED",  defaultSI: 0},
    21: { size: 4, name: "DateTime",  defaultSI: 0},
    22: { size: 5, name: "SV",  defaultSI: 0},
    23: { size: 5, name: "SVMin",  defaultSI: 0 },
    24: { size: 5, name: "SVMax",  defaultSI: 0},
    25: { size: 5, name: "TV",  defaultSI: 0},
    26: { size: 5, name: "TVMin",  defaultSI: 0 },
    27: { size: 5, name: "TVMax",  defaultSI: 0},
    28: { size: 5, name: "QV",  defaultSI: 0},
    29: { size: 5, name: "QVMin",  defaultSI: 0 },
    30: { size: 5, name: "QVMax",  defaultSI: 0},
  };

const floatFixedPoints = 3;

function Decode(bytes, variables) {
  
  let sensors = {};
  let i = 0;
  let isDeviceMalfunction = false;

  while (i < bytes.length) {

    let v_type = bytes[i++];
    
    if (typeof varTypes[v_type] == 'undefined'){
        break;
    }

    let v_size = varTypes[v_type].size;
    let v_name = varTypes[v_type].name;
    let v_si = varTypes[v_type].defaultSI;
    let raw = bytes.slice(i, v_size+i);

    switch (v_type){
      //Current
      //Percent
      //Battery
      case 0:
      case 7:
      case 12:
        v_siName = getSIName(v_si);
        v_value = Number(bytesToFloat(raw).toFixed(floatFixedPoints));
        sensors[v_name] = {value: v_value, si: v_siName, Raw: toHexString(raw)};
        break;
      //PV
      //PVMin
      //PVMax
      //SV
      //SVMin
      //SVMax
      //TV
      //TVMin
      //TVMax
      //QV
      //QVMin
      //QVMax
      case 1:
      case 4:
      case 5:
      case 22:
      case 23:
      case 24:
      case 25:
      case 26:
      case 27:
      case 28:
      case 29:
      case 30:
        v_siName = getSIName(raw[0]);
        v_value = Number(bytesToFloat(raw.slice(1, 5)).toFixed(floatFixedPoints));
        sensors[v_name] = {value: v_value, si: v_siName, Raw: toHexString(raw)};
        break;
      //Status
      case 2:
        v_value = bytesToDecimal(raw);
        let primaryVariableOutOfLimits = (v_value & (1 << 0)) ? true : false;
        let nonPrimaryVariableOutOfLimits = (v_value & (1 << 1)) ? true : false;
        let loopCurrentSaturated = (v_value & (1 << 2)) ? true : false;
        let loopCurrentFixed = (v_value & (1 << 3)) ? true : false;
        let moreStatusAvailable = (v_value & (1 << 4)) ? true : false;
        let coldStart = (v_value & (1 << 5)) ? true : false;
        let configurationChanged = (v_value & (1 << 6)) ? true : false;
        let deviceMalfunction = (v_value & (1 << 7)) ? true : false;
  
        sensors[v_name] = {
          PrimaryVariableOutOfLimits: primaryVariableOutOfLimits,
          NonPrimaryVariableOutOfLimits: nonPrimaryVariableOutOfLimits,
          LoopCurrentSaturated: loopCurrentSaturated,
          LoopCurrentFixed : loopCurrentFixed,
          MoreStatusAvailable: moreStatusAvailable,
          ColdStart: coldStart,
          ConfigurationChanged: configurationChanged,
          DeviceMalfunction: deviceMalfunction,
          Raw : toHexString(raw) };

          if (deviceMalfunction === true){
            isDeviceMalfunction = true;
          }
        break;
      //UID  
      case 3:
        v_value = bytesToDecimal(raw);
        sensors[v_name] = {value: v_value, Raw: toHexString(raw)};
        break;
      //MoreStatus
      case 6:
        sensors[v_name] = {Raw: toHexString(raw)};
        break;
      //UNDEF1-5
      case 8:
      case 9:
      case 10:
      case 11:
      case 13:
        break;
      //SelfState
      case 14:
        v_value = bytesToDecimal(raw);
        let errorFactorySettings = (v_value & (1 << 0)) ? true : false; 
        let errorReservedSettings = (v_value & (1 << 1)) ? true : false; 
        let errorUserSettings = (v_value & (1 << 2)) ? true : false; 
        let errorADC = (v_value & (1 << 3)) ? true : false; 
        let errorTransmitter = (v_value & (1 << 4)) ? true : false; 
        let isReedBeenActivated = (v_value & (1 << 5)) ? true : false;
        let errorArchive = (v_value & (1 << 6)) ? true : false;
        let errorFram = (v_value & (1 << 7)) ? true : false;
        let isThresholdMinActivated = (v_value & (1 << 8)) ? true : false;
        let isThresholdMaxActivated = (v_value & (1 << 9)) ? true : false;
  
        sensors[v_name] = {
            ErrorFactorySettings : errorFactorySettings, 
            ErrorReservedSettings : errorReservedSettings, 
            ErrorUserSettings : errorUserSettings, 
            ErrorADC : errorADC, 
            ErrorTransmitter : errorTransmitter, 
            IsReedBeenActivated : isReedBeenActivated,
            ErrorArchive : errorArchive,
            ErrorFram : errorFram,
            IsThresholdMinActivated: isThresholdMinActivated,
            IsThresholdMaxActivated: isThresholdMaxActivated,
            Raw : toHexString(raw),
            };
        break;
      //ConfGETBLOCK
      case 15:
        break;
      //DeviceInfo
      case 16:
        let versionFW =  bytesToDecimal(raw.slice(1, 2)).toString() + "." + bytesToDecimal(raw.slice(0, 1)).toString();
        let versionHW =  bytesToDecimal(raw.slice(3, 4)).toString() + "." + bytesToDecimal(raw.slice(2, 3)).toString();
        let crcFW = "0x" + bytesToDecimal(raw.slice(4, 8)).toString(16);
        let crcMetrolog =  "0x" + bytesToDecimal(raw.slice(8, 12)).toString(16);
        let measureMethod = raw[12];
        switch (measureMethod){
          case 0:
            measureMethod = "LOOP";
            break;
          case 1:
            measureMethod = "HART";
            break;
          case 2:
            measureMethod = "SWIRE";
            break;
        }

        sensors[v_name] = {
          VersionFW : versionFW,
          VersionHW : versionHW,
          CRC_FW: crcFW,
          CRC_Metrolog: crcMetrolog,
          MeasureMethod : measureMethod,
          Raw : toHexString(raw),
        };
        break;
      //Threshold
      case 17:
        let initialValue = raw[0];
        switch (initialValue){
          case 0:
            initialValue = "Current";
            break;
          case 1:
            initialValue = "PV";
            break;
          case 2:
            initialValue = "Percent";
            break;
        }
        
        let thresholdMin  = Number(bytesToFloat(raw.slice(1, 5)).toFixed(floatFixedPoints));
        let thresholdMax  = Number(bytesToFloat(raw.slice(5, 9)).toFixed(floatFixedPoints));
        let thresholdHyst = Number(bytesToFloat(raw.slice(9, 13)).toFixed(floatFixedPoints));
        let thresholdEnMin = (raw[13] & (1 << 0)) ? true : false; 
        let thresholdEnMax = (raw[14] & (1 << 0)) ? true : false;  
        
        sensors[v_name] = {
          InitialValue : initialValue,
          ThresholdMin : thresholdMin,
          ThresholdMax : thresholdMax,
          ThresholdHyst : thresholdHyst,
          ThresholdEnMin : thresholdEnMin,
          ThresholdEnMax : thresholdEnMax,
          Raw: toHexString(raw)
        };
        break;
      //Measure
      case 18:
        let sendVar = raw[0];
        switch (sendVar){
          case 0:
            sendVar = "Source";
            break;
          case 1:
            sendVar = "Pv & Percent";
            break;
        }
        let warmUpDelay =  raw[1];
        let sendPeriodMins =  raw[2];
        let measurePeriodMins =  raw[3];
        
        sensors[v_name] = {
          SendVar : sendVar,
          WarmUpDelay : warmUpDelay,
          SendPeriodMins : sendPeriodMins,
          MeasurePeriodMins : measurePeriodMins,
          Raw: toHexString(raw)
        };
        break;
      //Lora
      case 19:
        let retransmissionCount = raw[0];
        let dr = raw[1];
        let txPower = raw[2];
        let isLBTEnable = (raw[3] & (1 << 0)) ? true : false; 
        
        let sign = raw[5] & (1 << 7);
        let lbtRSSI = (((raw[5] & 0xFF) << 8) | (raw[4] & 0xFF));
        if (sign) {
          lbtRSSI = 0xFFFF0000 | lbtRSSI;  
        }
        let lbtScanTime = raw[6];
        
        sensors[v_name] = {
          RetransmissionCount : retransmissionCount,
          DR : dr,
          TxPower: txPower,
          IsLBTEnable : isLBTEnable,
          LBTRSSI : lbtRSSI,
          LBTScanTime : lbtScanTime,
          Raw: toHexString(raw)
        };
        break;
      //Reed
      case 20:
        let isReedActive = (raw[0] & (1 << 0)) ? true : false; 
        let reedWarmUpDelay =  raw[1]; 
  
        sensors[v_name] = {
          IsReedActive : isReedActive,
          ReedWarmUpDelay : reedWarmUpDelay,
          Raw: toHexString(raw),
        };
        break;
    }  

    i += v_size;
  }

  if (isDeviceMalfunction === true){
    if (sensors.hasOwnProperty("Current")){
      v_siName = getSIName(varTypes[7].defaultSI);
      v_value = ((sensors["Current"].value - 4.0 ) / 16.0) * 100.0;
      sensors["Percent"] = { value: v_value, si: v_siName};
  
      if (typeof variables.rangeMax != "undefined"){
        if (typeof variables.rangeMin != "undefined"){
          if (typeof variables.rangeSi != "undefined"){
  	          v_siName = getSIName(variables.rangeSi);
  	          v_value = (sensors["Percent"].value / 100.0 ) * (variables.rangeMax - variables.rangeMin)
  	          sensors["PV"] = { value: v_value, si: v_siName};
          }
        }
      }
    }
  }
  return sensors;
}

function getSIName(si) {
  if (typeof siTypes[si] == 'undefined'){
      return siTypes[0].name;
  }
  return siTypes[si].name;
}

function bytesToDecimal(bytes) {
  var d = 0;
  for (var i = bytes.length - 1; i >= 0; i--) {
    d = d * 256 + bytes[i];
  }
  return d;
}

function bytesToFloat(bytes) {
  var bits = (bytes[3] << 24) | (bytes[2] << 16) | (bytes[1] << 8) | bytes[0];
  var sign = bits >>> 31 === 0 ? 1.0 : -1.0;
  var e = (bits >>> 23) & 0xff;
  var m = e === 0 ? (bits & 0x7fffff) << 1 : (bits & 0x7fffff) | 0x800000;
  var f = sign * m * Math.pow(2, e - 150);
  return f;
}

function toHexString(byteArray) {
  return Array.from(byteArray, function(byte) {
    return ('0' + (byte & 0xFF).toString(16)).slice(-2);
  }).join('')
}
// Encode downlink function.
//
// Input is an object with the following fields:
// - data = Object representing the payload that must be encoded.
// - variables = Object containing the configured device variables.
//
// Output must be an object with the following fields:
// - bytes = Byte array containing the downlink payload.
function encodeDownlink(input) {
  return {
    bytes: [225, 230, 255, 0]
  };
}
